//
//  fcDio32.h
//
//  header file for the FPCI-DIO32 related library calls in the fcFw framework.
//
//
//  Created by Bob Piatek on Tue Aug 03 2004.
//
////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2001 - 2004 fishcamp engineering. All rights reserved.
//
// fishcamp engineering provides this AS IS, WITHOUT ANY WARRANTY, EXPRESS OR
// IMPLIED, INCLUDING, WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY
// OR FITNESS FOR A PARTICULAR PURPOSE.  fishcamp engineering makes no guarantee
// or representations regarding the use of, or the results of the use of,
// the software and documentation in terms of correctness, accuracy,
// reliability, currentness, or otherwise; and you rely on the software,
// documentation and results solely at your own risk.
// 
// IN NO EVENT SHALL FISHCAMP ENGINEERING BE LIABLE FOR ANY LOSS OF USE, LOSS 
// OF BUSINESS, LOSS OF PROFITS, INDIRECT, INCIDENTAL, SPECIAL OR CONSEQUENTIAL
// DAMAGES OF ANY KIND.  IN NO EVENT SHALL FISHCAMP ENGINEERING'S TOTAL LIABILITY 
// EXCEED THE SUM PAID TO FISHCAMP ENGINEERING FOR THE PRODUCT LICENSED HEREUNDER.
// 
////////////////////////////////////////////////////////////////////////


#import <Foundation/Foundation.h>
#include <IOKit/IOKitLib.h>
#include "fpciDio32xUserClient.h"



// constants
// card's registers in the Xilinx FPGA

#define	c_dio32_mem_bk_size		0x00800000
#define	c_dio32_mem_bk0			0x00000000
#define	c_dio32_mem_bk1			0x01000000
#define	c_dio32_bk2_data		0x02000000
#define	c_dio32_dds_freq		0x02000004
#define	c_dio32_dds_cntrl		0x02000008
#define	c_dio32_master_cntrl	0x0200000C
#define	c_dio32_mode0_data		0x02000010
#define	c_dio32_mode0_cntrl		0x02000014
#define	c_dio32_mode1_cntrl		0x02000018
#define	c_dio32_match_pattern	0x0200001C
#define	c_dio32_pattern_mask	0x02000020


// error codes
#define	fc_noErr					0
#define	fc_Err						-1
#define	fc_ErrWaveformMemFull		-2
#define	fc_ErrMemoryPageBoundary	-3
#define	fc_ErrMemory				-4







// struct describes a single data pattern word represented by a single clock cycle of the output 
// pattern generator on the FPCI-DIO32 card.

typedef struct wvPatternWord {
	UInt8   flags;
	UInt8   dirCntrl;
	UInt32  patternData;
} wvPatternWord;


// this structure will define the individual waveform segments in the output pattern memory array

typedef struct wvSegRecord {
	UInt32			wv_this_segment_address;
	UInt32			wv_next_segment_address;
	UInt16			wv_num_samples;
	UInt16			wv_repeat_count;
	UInt32			wv_reserved;
	NSMutableData   *pattern_data;
} wvSegRecord;





@interface fcDio32 : NSObject {

	NSMutableArray		*waveform;				// array of NSMutableData objects each representing 
												// individual wvSegRecord
												
	UInt32				DDS_Frequency;			// stores the currrent frequency setting of the DDS chip

	// variables needed when talking to the FPCI-DIO32 card
	mach_port_t			gMasterPort;
	io_iterator_t		gCards;
	io_registry_entry_t	gCard;
	io_connect_t		gConnect;				// reference to our card
	void				*gPCIMemAddress;		// used for memory window to the card from user space
	unsigned int		gPCIMemSize;			// 64MByte memory region
	
	
	NSMutableString*	logText;				// we keep a local log of status and error messages


}


- (id)init;
- (void)dealloc;

// waveform model maintenance routines
// these routines will manage the waveform storage.  Routines exist to manage waveform segment records.
//
//


// return the number of waveform segments currently defined in the waveform
- (int) getNumDefinedWaveformSegments;



// this routine will delete any currently defined waveform segments in the 'waveform' array.
- (void)clearWaveformDefinition;	
	


// allocate and add a new wvSegRecord and insert it at the end of the waveforem array.
// mark it as the last segment in the waveform and link it to the previous last waveform segment in the waveform.
- (void)newWvSegRecord;		


// delete a waveform segment from the array
-(void) deleteWaveformSegment: (int) segmentNumber;



// add another 'wvPatternWord' to the 'wvSegRecord' at 'segmentIndex' in the 'waveform' array
// segmentIndex = 0 .. n
- (void)addPatternWord2Segment: (wvPatternWord *) patternWord atIndex: (int) segmentIndex;		

// add many new 'wvPatternWord' to the 'wvSegRecord' at 'segmentIndex' in the 'waveform' array
// segmentIndex = 0 .. n
// the pattern words are encapsulated in the NSData* object
- (void)addManyPatternWord2Segment: (NSData *) patternWords atIndex: (int) segmentIndex;		


// delete a pattern word in a waveform segment from the array
-(void) deletePatternWordFromWaveformSegment: (int) segmentNumberIndex thePatternWord: (int) patternWordIndex;



// simply return a pointer to the data for the specified waveform segment record.  
// segmentIndex = 0 .. n
- (wvSegRecord *)getWaveSegmentRecordAtIndex: (int) segmentIndex;


// get the input words that were captured for the given segmentIndex
// segmentIndex = 0 .. n
// will return the data bytes (4 per word) in the NSData* object
- (NSData*) getInputWaveformWordsForIndex:  (int) segmentIndex;


// waveform consistancy check routine
- (int) waveformConsistancyCheck;





// the following routine will create and return an NSData object that will essentially contain the 
// waveform definition as a byte stream.  Other header information is added to make this routine suitable
// for use during storage operations to disk.
 - (NSData*) getWaveformDefinitionAsNSData;
 
 
 
 
 
 // the following routine will take a byte streem as an NSData object and load the waveform definition array from the data.
 -(void) loadWaveformDefinitionFromNSData: (NSData*) theRawData;






//
// routine to prepare for display the waveform records as they are currently defined in memory
// this routine will return an NSMutableArray* object that will contain X entries.
// Each entry will be an NSDictionary* object.  Each dictionary entry will contain an NSString*.
// GUI software can  walk through the array to display the HEX waveform data.
// the following NSDictionary keys are used:
//   For HEADER Data:
//     @"Segment_num"
//     @"wv_this_segment_address"
//     @"wv_next_segment_address"
//     @"wv_repeat_count"
//     @"wv_num_samples"
//     @"wv_reserved"
//   For PATTERN Data:
//     @"clkNum"
//     @"flags"
//     @"dirCntrl"
//     @"outPatternData"
//     @"inPatternData"
//
// the waveform data will be formatted as HEX data.
//
// This routine only examines the memory on the card itself so is a true representation of what
// the card will execute.
//
// Be carefull.  this routine can take a long time to execute for large waveforms.
//
// waveform segments with REPEAT counts > 1 are not represented as such.
//
- (NSMutableArray*) prepareForDisplayWaveformMemoryDump;



//
// routine to prepare for display the waveform records as they are currently defined before download to the card
// this routine will return an NSMutableArray* object that will contain X entries.
// Each entry will be an NSDictionary* object.  Each dictionary entry will contain an NSString*.
// GUI software can  walk through the array to display the HEX waveform data.
// the following NSDictionary keys are used:
//   For HEADER Data:
//     @"Segment_num"
//     @"wv_this_segment_address"
//     @"wv_next_segment_address"
//     @"wv_repeat_count"
//     @"wv_num_samples"
//     @"wv_reserved"
//   For PATTERN Data:
//     @"clkNum"
//     @"flags"
//     @"dirCntrl"
//     @"outPatternData"
//
// the waveform data will be formatted as HEX data.
//
// This routine displays the contects of the 'waveform' memory before it is downloaded to the card.
//
// Be carefull.  this routine can take a long time to execute for large waveforms.
//
// waveform segments with REPEAT counts > 1 are not represented as such.
//
- (NSMutableArray*) prepareForDisplayWaveformDefinition;





// interface to the hardware

// This routine will open up a connection to our PCI card.
// call this routine once at the beginning of your application.
// Applications may wish to call 'InitCard' instead of this routine
// since it does a more thorough initialization.
- (kern_return_t) OpenCardConnection;

// this routine will clean up after we are done talking to our card.
// call this routine before your application quits.
- (void) CloseCardConnection;

// Call this once to get a connection to our card and init some registers to default.
// This routine calls 'OpenCardConnection' and then does additionaal initialization.
// this routine is called automatically by the 'init' method 
- (kern_return_t) InitCard;

// helper routine allows us to get the current setting of the 'DDS Frequency' register on the card.
- (kern_return_t) fpciDio32x_GetDDS_Freq_Reg: (UInt32 *) value;

// helper routine allows us to set the value of the 'DDS Freq' on the card.
// newFrequency = frequency in Hertz
- (kern_return_t) fpciDio32x_SetDDS_Frequency: (UInt32) newFrequency;

// helper routine allows us to set the value of the 'DDS Freq' register on the card.
- (kern_return_t) fpciDio32x_SetDDS_Freq_Reg: (UInt32) data;

// helper routine allows us to get the current setting of the 'DDS Cntrl' register on the card.
- (kern_return_t) fpciDio32x_GetDDS_Cntrl_Reg: (UInt32 *) value;

// helper routine allows us to set the value of the 'DDS Cntrl' register on the card.
- (kern_return_t) fpciDio32x_SetDDS_Cntrl_Reg: (UInt32) data;

// helper routine allows us to get the current setting of the 'Master Cntrl' register on the card.
- (kern_return_t) fpciDio32x_GetMaster_Cntrl_Reg: (UInt32 *) value;

// helper routine allows us to set the value of the 'Master Cntrl' register on the card.
- (kern_return_t) fpciDio32x_SetMaster_Cntrl_Reg: (UInt32) data;

// helper routine allows us to get the current setting of the 'Mode0 Data' register on the card.
- (kern_return_t) fpciDio32x_GetMode0_Data_Reg: (UInt32 *) value;

// helper routine allows us to set the value of the 'Mode0 Data' register on the card.
- (kern_return_t) fpciDio32x_SetMode0_Data_Reg: (UInt32) data;

// helper routine allows us to get the current setting of the 'Mode0 Cntrl' register on the card.
- (kern_return_t) fpciDio32x_GetMode0_Cntrl_Reg: (UInt32 *) value;

// helper routine allows us to set the value of the 'Mode0 Cntrl' register on the card.
- (kern_return_t) fpciDio32x_SetMode0_Cntrl_Reg: (UInt32) data;

// helper routine allows us to get the current setting of the 'Mode1 Cntrl' register on the card.
- (kern_return_t) fpciDio32x_GetMode1_Cntrl_Reg: (UInt32 *) value;

// helper routine allows us to set the value of the 'Mode1 Cntrl' register on the card.
- (kern_return_t) fpciDio32x_SetMode1_Cntrl_Reg: (UInt32) data;

// helper routine allows us to get the current setting of the 'c_match_pattern' register on the card.
- (kern_return_t) fpciDio32x_GetMatch_pattern_Reg: (UInt32 *) value;

// helper routine allows us to set the value of the 'c_match_pattern' register on the card.
- (kern_return_t) fpciDio32x_SetMatch_pattern_Reg: (UInt32) data;

// helper routine allows us to get the current setting of the 'c_pattern_mask' register on the card.
- (kern_return_t) fpciDio32x_GetPattern_mask_Reg: (UInt32 *) value;

// helper routine allows us to set the value of the 'c_pattern_mask' register on the card.
- (kern_return_t) fpciDio32x_SetPattern_mask_Reg: (UInt32) data;

// helper routine allows us to get the current value of a memory location in the 'input' memory array.
- (kern_return_t) fpciDio32x_GetInputMemoryAtAddress: (UInt32) address theData: (UInt32 *) data;

// helper routine allows us to set the value of a memory location in the 'input' memory array.
- (kern_return_t) fpciDio32x_SetInputMemoryAtAddress: (UInt32) address theData: (UInt32) data;

// helper routine allows us to get the current value of a memory location.
- (kern_return_t) fpciDio32x_GetOutputMemoryAtAddress: (UInt32) address theData: (UInt32 *) dataValue theCntrl: (UInt16 *)cntrlValue;

// helper routine allows us to set the value of a memory location.
- (kern_return_t) fpciDio32x_SetOutputMemoryAtAddress: (UInt32) address theData: (UInt32) dataValue theCntrl: (UInt16) cntrlValue;

- (BOOL) doFpciDio32xDirectMemoryTests;

- (BOOL) doFpciDio32xMemoryTests;

- (void)uploadWaveformData2Hardware;	// tell the hardware what the waveform looks like


- (void)stopMode1;
- (void)startMode1;





// waveform pattern generation routines.
// these routines will allow creation of various pattern words and sequences of pattern words
// that can be stored in the waveform data structure of the card.  Use the waveforem model maintenance
// routines to store the pattern word after creation.
// the user can build up any waveforem by calling these routines.
//
//
//




// helper routine that will return a bit mask given the specified bit lane
// bit lanes are specified from 31 (MSB) to 0 (LSB)
- (UInt32)getLaneBitMask: (UInt8) theBitLane;


// helper routine that will return a bit mask to be used when addressing the direction
// control bit given the specified bit lane
// bit lanes are specified from 31 (MSB) to 0 (LSB)
- (UInt8)getNibbleLaneDirMask: (UInt32) theBitLane;



// given an initial pattern word, this routine will return an NSData* object that contains the
// pattern replicated a desired number of times.  This can be usefull for capturing long sequences
// of input data to the input memory array.
// specify the number of times to replicate the pattern in the 'count' parameter.  The 'count' must
// be between 1 and 252 which is the max number of pattern words that may fit in a waveform segment
// record and still fit within a RAM page.
- (NSData*)replicatePatternWithInitialPattern: (wvPatternWord *) initialPatternData numTimes: (UInt8) count;





// generate a pulse on a bit lane
// Specify:
//		initialPatternData	-	the initial pattern word
//		signalLane			-	the bit lane that the pulse will be generated on
//		theState			-	the level of the signal during the pulse
//		beforeCount			-	the number of clock cycles before the pulse begins
//		durationCount		-	the pulse width in clock cycles
//		afterCount			-	the number of clock cycles after the pulse ends
//
// the counts may be zero but the total number of clock cycles should be less than 252 in
// order to ensure that the waveform segment being generated fits within a RAM page.
//
- (NSData*)genPulseWithInitialPattern:			(wvPatternWord *) initialPatternData
							theSignalLane:		(UInt8) signalLane
							theState:			(BOOL)	theState
							clocksBefore:		(UInt8) beforeCount
							clocksDuration:		(UInt8) durationCount
							clocksAfter:		(UInt8) afterCount;



// set the desired state of the specified signal lane
- (NSData*)setSignalStateWithInitialPattern:		(wvPatternWord *) initialPatternData
									theState:		(BOOL)   theState
									theSignalLane:	(UInt8)  theSignalLane;





// the following routines are usefull for simulating an IIC serial data bus.
// Two signals lines are used to specify the SCLK and SDATA signals on the IIC interface
// These signal lines are specified with their bit position in the 32 bit word
// from 31 (MSB) to 0 (LSB)
//





// generate the bus idle sequence on the IIC bus for 'count' clock cycles.
- (NSData*)writeIICBusIdleWithInitialPattern:		(wvPatternWord *) initialPatternData
									clocksDuration:	(UInt8) count
									theSCLKLane:	(UInt8) SCLKLane
									theSDATALane:   (UInt8) SDATALane;


// generate the pattern sequence to write to a specified register in the IIC device
// You pass in the register#, data, and the initial pattern state and get back the sequence of
// wvPatternWord that perform the operation.  The bit lanes to use for the SCLK and SDATA lines are specified.
- (NSData*)writeIICRegister:						 (UInt8)  theReg 
									theSlaveAddress: (UInt8) theAddress
									theData:		 (UInt16) data
									initialPattern:  (wvPatternWord *) initialPatternData
									theSCLKLane:	 (UInt8)  SCLKLane
									theSDATALane:    (UInt8)  SDATALane;

// write the IIC Start bit to the IIC device
- (NSData*)writeIICStartBitWithInitialPattern:		(wvPatternWord *) initialPatternData
									theSCLKLane:	(UInt8)  SCLKLane
									theSDATALane:	(UInt8)  SDATALane;

// write the IIC Stop bit to the IIC device
- (NSData*)writeIICStopBitWithInitialPattern:		(wvPatternWord *) initialPatternData
									theSCLKLane:	(UInt8)  SCLKLane
									theSDATALane:   (UInt8)  SDATALane;

// get the IIC ACK bit from the IIC device
- (NSData*)getIICACKBitWithInitialPattern:			(wvPatternWord *) initialPatternData
									theSCLKLane:	(UInt8)  SCLKLane
									theSDATALane:   (UInt8)  SDATALane;

// write a byte to the device on the IIC bus
- (NSData*)writeIICByte:							(UInt8) theByte
								WithInitialPattern: (wvPatternWord *) initialPatternData
								theSCLKLane:		(UInt8)  SCLKLane
								theSDATALane:		(UInt8)  SDATALane;

// write an IIC data bit to the IIC device
- (NSData*)writeIICBit:								(BOOL) theBit
								WithInitialPattern: (wvPatternWord *) initialPatternData
								theSCLKLane:		(UInt8)  SCLKLane
								theSDATALane:		(UInt8)  SDATALane;




// this routine will add another log message to the stuff already there
- (void) addLogMessage: (NSString*) newMessage withTimeStamp: (BOOL) timeFlag;


- (NSMutableString *)logText;
- (void)setLogText:(NSMutableString *)newLogText;







@end
